
/*******************************************************************************
*                                                                            
*                     TWO DIMENSIONAL STATIC ELASTICITY 
*
*                                  C++ Code
*                            Version: January 2009
*      Author: Arash Zamani, PhD from Amirkabir University of Technology
*        
*           This code has been tested with Microsoft Visual C++ 2008
********************************************************************************
*                                                                            
*                       Introducing global variables                         
*                                                                            
* nx=number of divisions in x direction                                    
* ny=number of divisions in y direction                                   
* npe=number of nodes per element                                                      
* nb=bandwith                                                               
* ne=number of elements                                                     
* nn=number of nodes                                                        
* ndf=number of degrees of freedom per node                                 
* ndfe=number of degrees of freedom per element                             
* tndf=total number of degrees of freedom                                   
* node=connectivity                                                          
* xv=vector of x values                                                     
* yv=vector of y values                                                      
* const1=determines whether to impose primary boundary conditions       
* const2=determines the value of primary boundary conditions                
* gk=global stiffness matrix                                                
* p=the vector which is first used to store external forces and then         
*   to store the calculated primary variables from finite element model.      
* landa,G=Lame constants                                                    
* ro=density                                                                
*                                                                            
******************************************************************************/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <io.h>
#include <iostream>
#include <fstream>
#include <float.h>
using namespace std;

//four node rectangular element (LRE)
const long  nx=24,ny=6,npe=4,ndf=2;                   
const long  nb=ndf*(nx+3),nn=(nx+1)*(ny+1);            

//eight node rectangular element (serendipity)
//const long  nx=12,ny=2,npe=8,ndf=2;                     
//const long  nb=ndf*(3*nx+5),nn=(3*nx+2)*(ny+1)-nx-1;  

const long  ne=nx*ny,tndf=ndf*nn,ndfe=npe*ndf;
long        node[npe*ne+1],const1[tndf+1];
double      xv[nn+1], yv[nn+1],const2[tndf+1];
double      gk[tndf*nb+1],p[tndf+1];
double      G=77.0*pow(10,9),landa=114.0*pow(10,9),ro=7820.0;


void mesh_LRE(double x0,double y0);
void mesh_serendipity(double x0,double y0);
void core_LRE();
void core_serendipity();
void boundary_sym_banded(long neqns,long nbw,double mat[],double rhs[],
						 long cond1[],double cond2[]);
void solve_gauss_sym_banded(long neqns,long nbw,double mat[],double rhs[]);
void file_out();




void main(void)
{

   mesh_LRE(6.0,1.0);
   core_LRE();
   boundary_sym_banded(tndf,nb,gk,p,const1,const2);
   solve_gauss_sym_banded(tndf,nb,gk,p);

   file_out();
}

void mesh_LRE(double x0,double y0)//generates the mesh for a rectangular 
                                  //domain by four node rectangular elements
{
    long i,j,n,m,nx1,ny1;
    double f;

    nx1=nx+1;
    ny1=ny+1;
    m=-4;
    for (j=1;j<=ny;j++){
        for (i=1;i<=nx;i++){

            m+=4;
            n=nx1*(j-1)+i;

            node[m+1]=n;
            node[m+2]=n+1;
            node[m+3]=n+nx+2;
            node[m+4]=n+nx+1;
		}
	}

    for (i=1;i<=nx1;i++){
        for (j=1;j<=ny1;j++){
			n=nx1*(j-1)+i;
			xv[n]=double(i-1)*x0/double(nx);
			yv[n]=double(j-1)*y0/double(ny);
		}
	}

	for (i=1;i<=tndf;i++){
        const1[i]=0;
    }

    for (j=1;j<=ny1;j++){
        n=nx1*(j-1)+1;

        const1[ndf*(n-1)+1]=1;
        const2[ndf*(n-1)+1]=0.0;

        const1[ndf*(n-1)+2]=1;
        const2[ndf*(n-1)+2]=0.0;
	}

    f=-1.0/double(ny);
    for (j=2;j<=ny;j++){
        n=nx1*j;
		p[ndf*(n-1)+2]=f;
    }
    p[ndf*(nx1-1)+2]=f/2.0;
    p[ndf*(nx1*ny1-1)+2]=f/2.0;
}

void mesh_serendipity(double x0,double y0)//generates the mesh for a rectangular 
                                          //domain by eight node rectangular 
										  //elements
{
    long   i,j,n,m,nx1,ny1;
    double f,dx,dy;

	nx1=nx+1;
	ny1=ny+1;
   
    dx=x0/double(2*nx);
    dy=y0/double(2*ny);

    m=-8;
	for (j=1;j<=ny;j++){
        for (i=1;i<=nx;i++){

		    m+=8;	
		    n=(3*nx+2)*(j-1)+2*i-1;
            
			node[m+1]=n;
            node[m+2]=n+1;
		    node[m+3]=n+2;
			node[m+4]=n+2*nx+3-i;
            node[m+5]=n+3*nx+4;
			node[m+6]=node[m+5]-1;
            node[m+7]=node[m+5]-2;
			node[m+8]=node[m+4]-1;
		}
	}

	for (i=1;i<=nx1;i++){
		for (j=1;j<=ny1;j++){
			n=(3*nx+2)*(j-1)+2*i-1;
			xv[n]=double(2*i-2)*dx;
			yv[n]=double(2*j-2)*dy;
		}
	}

	for (i=1;i<=nx;i++){
		for (j=1;j<=ny1;j++){
			n=(3*nx+2)*(j-1)+2*i;
			xv[n]=double(2*i-1)*dx;
			yv[n]=double(2*j-2)*dy;
		}
	}

	for (i=1;i<=nx1;i++){
        for (j=1;j<=ny;j++){
			n=(3*nx+2)*(j-1)+2*nx+1+i;
			xv[n]=double(2*i-2)*dx;
			yv[n]=double(2*j-1)*dy;
		}
	}
	
	for (i=1;i<=tndf;i++){
        const1[i]=0;
    }
    
    for (j=1;j<=ny1;j++){
        n=(3*nx+2)*(j-1)+1;

        const1[ndf*(n-1)+1]=1;
        const2[ndf*(n-1)+1]=0.0;

	    const1[ndf*(n-1)+2]=1;
        const2[ndf*(n-1)+2]=0.0;
	}

	for (j=1;j<=ny;j++){
        n=(3*nx+2)*(j-1)+(2*nx+1)+1;

        const1[ndf*(n-1)+1]=1;
        const2[ndf*(n-1)+1]=0.0;

	    const1[ndf*(n-1)+2]=1;
        const2[ndf*(n-1)+2]=0.0;
	}

	f=-1.0/double(2*ny);
    for(j=2;j<=ny;j++){

        n=(3*nx+2)*(j-1)+2*nx+1;
        p[ndf*(n-1)+2]=f;

        n=(3*nx+2)*j;
        p[ndf*(n-1)+2]=f;
	}
    n=2*nx+1;
    p[ndf*(n-1)+2]=f/2.0;

    n=3*nx+2;
    p[ndf*(n-1)+2]=f;

    n=(3*nx+2)*ny+2*nx+1;
    p[ndf*(n-1)+2]=f/2.0;

}

void core_LRE()//calculates element matrices and assembles them in 
               //the global matrix for four node rectangular elements
{
   long   i,j,ii,jj,ig,jg,r1,r2,num;
   double c00,c01,c02,c11,c12,c22,a,b;
   long   n[npe+1];
   double x[npe+1],y[npe+1];
   double k11[npe+1][npe+1],k12[npe+1][npe+1],k22[npe+1][npe+1];
   double k1[ndfe+1][ndfe+1],p1[ndfe+1];

   double S11[4][4]={{2,-2,-1,1},{-2,2,1,-1},{-1,1,2,-2},{1,-1,-2,2}};
   double S12[4][4]={{1,1,-1,-1},{-1,-1,1,1},{-1,-1,1,1},{1,1,-1,-1}};
   double S22[4][4]={{2,1,-1,-2},{1,2,-2,-1},{-1,-2,2,1},{-2,-1,1,2}};


   for (num=1;num<=ne;num++){

       ii=npe*(num-1);
       for (j=1;j<=npe;j++){

           n[j]=node[ii+j];
           x[j]=xv[n[j]];
           y[j]=yv[n[j]];
       }

       a=x[2]-x[1];
       b=y[4]-y[1];


       c00=a*b/36.0;
       c01=b/12.0;
       c02=a/12.0;
       c11=b/6.0/a;
       c12=1.0/4.0;
       c22=a/6.0/b;

       for (i=1;i<=npe;i++){
           for (j=1;j<=npe;j++){

               k11[i][j]=(2.0*G+landa)*c11*S11[i-1][j-1]+G*c22*S22[i-1][j-1];
               k12[i][j]=G*c12*S12[j-1][i-1]+landa*c12*S12[i-1][j-1];
			   k22[i][j]=(2.0*G+landa)*c22*S22[i-1][j-1]+G*c11*S11[i-1][j-1];
		   }
	   }


       ii=1;
       for (i=1;i<=npe;i++){
           jj=1;
           for (j=1;j<=npe;j++){
               
			   k1[ii][jj]=k11[i][j];
               k1[ii][jj+1]=k12[i][j];
			   k1[ii+1][jj]=k12[j][i];
               k1[ii+1][jj+1]=k22[i][j];
			   jj+=ndf;
		   }
		   ii+=ndf;
	   }
 
       
	   for (i=1;i<=ndfe;i++) p1[i]=0.0;

       for (i=1;i<=npe;i++){
           ig=(n[i]-1)*ndf;
           r1=(i-1)*ndf;
           for (ii=1;ii<=ndf;ii++){
               ig++;r1++;
               p[ig]+=p1[r1];
               for (j=1;j<=npe;j++){
                    jg=(n[j]-1)*ndf;
                    r2=(j-1)*ndf;
                    for (jj=1;jj<=ndf;jj++){
                        jg++;r2++;
                        if(jg>=ig)     gk[(ig-1)*nb+jg-ig+1]+=k1[r1][r2];
                    }
               }
		   }
	   }
   }

}

void core_serendipity()//calculates element matrices and assembles them in 
                       //the global matrix for eight node rectangular elements
{
   long   i,j,ii,jj,ig,jg,r1,r2,num;
   double c00,c01,c02,c11,c12,c22,a,b;
   long   n[npe+1];
   double x[npe+1],y[npe+1];
   double k11[npe+1][npe+1],k12[npe+1][npe+1],k22[npe+1][npe+1];
   double k1[ndfe+1][ndfe+1],p1[ndfe+1];
 
   double S11[8][8]={{52, -80, 28, -6, 23, -40,17, 6},
                     {-80, 160, -80, 0, -40, 80, -40, 0},
                     {28, -80,52, 6, 17, -40, 23, -6},
                     {-6, 0, 6, 48, 6, 0, -6, -48},
                     {23, -40, 17, 6, 52, -80, 28, -6},
                     {-40, 80, -40, 0, -80, 160, -80, 0},
                     {17, -40, 23, -6, 28, -80, 52, 6},
                     {6, 0, -6, -48, -6, 0, 6, 48}};

   double S12[8][8]={{17, 4, -3, -4, 7, -4, 3, -20},
                     {-20, 0, 20, -16, -4, 0, 4, 16},
                     {3, -4, -17, 20, -3, 4, -7, 4},
                     {-4, -16, -4, 0, 4, 16, 4, 0},
                     {7, -4, 3, -20, 17, 4, -3, -4},
                     {-4, 0, 4, 16, -20, 0, 20, -16},
                     {-3, 4, -7, 4, 3, -4, -17, 20},
                     {4, 16, 4, 0, -4, -16, -4, 0}};

   double S22[8][8]={{52, 6, 17, -40, 23, -6, 28, -80},
                     {6, 48, 6, 0, -6, -48, -6, 0},
                     {17, 6, 52, -80, 28, -6, 23, -40},
                     {-40, 0, -80, 160, -80, 0, -40, 80},
                     {23, -6, 28, -80, 52, 6, 17, -40},
                     {-6, -48, -6, 0, 6, 48, 6, 0},
                     {28, -6, 23, -40, 17, 6, 52, -80},
                     {-80, 0, -40, 80, -40, 0, -80, 160}};
     
     
     
   for (num=1;num<=ne;num++){

       ii=npe*(num-1);
       for (j=1;j<=npe;j++){

           n[j]=node[ii+j];
           x[j]=xv[n[j]];
           y[j]=yv[n[j]];
       }

       a=x[3]-x[1];
       b=y[7]-y[1];

       c00=a*b/45.0;
       c01=b/90.0;
       c02=a/90.0;
       c11=b/a/90.0;
       c22=a/b/90.0;
       c12=1.0/36.0;

       for (i=1;i<=npe;i++){
           for (j=1;j<=npe;j++){

               k11[i][j]=(2.0*G+landa)*c11*S11[i-1][j-1]+G*c22*S22[i-1][j-1];
               k22[i][j]=(2.0*G+landa)*c22*S22[i-1][j-1]+G*c11*S11[i-1][j-1];
               k12[i][j]=G*c12*S12[j-1][i-1]+landa*c12*S12[i-1][j-1];
		   }
	   }


       ii=1;
       for (i=1;i<=npe;i++){
           jj=1;
           for (j=1;j<=npe;j++){

               k1[ii][jj]=k11[i][j];
               k1[ii][jj+1]=k12[i][j];
			   k1[ii+1][jj]=k12[j][i];
               k1[ii+1][jj+1]=k22[i][j];
               jj+=ndf;
           }
           ii+=ndf;
	   }


       for (i=1;i<=ndfe;i++)  p1[i]=0;

       for (i=1;i<=npe;i++){
           ig=(n[i]-1)*ndf;
           r1=(i-1)*ndf;
           for (ii=1;ii<=ndf;ii++){
               ig++;r1++;
               p[ig]+=p1[r1];
               for (j=1;j<=npe;j++){
                    jg=(n[j]-1)*ndf;
                    r2=(j-1)*ndf;
                    for (jj=1;jj<=ndf;jj++){
                        jg++;r2++;
                        if(jg>=ig)     gk[(ig-1)*nb+jg-ig+1]+=k1[r1][r2];
					}
			   }
		   }
	   }
	}

}

void boundary_sym_banded(long neqns,long nbw,double mat[],double rhs[],
						 long cond1[],double cond2[])
//imposes the boundary conditions on primary variables for
//a banded symmetric form
{
	long ii,i1,i2,i3,j;
	
	for (ii=1;ii<=neqns;ii++){
		
		if (cond1[ii]!=0){
	        i1=ii;
			i2=ii;
			i3=(ii-1)*nbw;
			for (j=2;j<=nbw;j++){
		       i1--;
		       i2++;
			   if(i1>=1){
				   rhs[i1]-=mat[(i1-1)*nbw+j]*cond2[ii];
				   mat[(i1-1)*nbw+j]=0.0;
			   }
			   if(i2<=neqns){
				   rhs[i2]-=mat[i3+j]*cond2[ii];
				   mat[i3+j]=0.0;
			   }
			}
	  
			rhs[ii]=cond2[ii];
	        mat[i3+1]=1.0;
		}
	}

}

void solve_gauss_sym_banded(long neqns,long nbw,double mat[],double rhs[])
//solves the system of equations in banded symmetric form by
//Gauss elimination method 
{

    long        i,j,ii,jj,m,n,n1,n2,n3;
    double      factor;
  
	m=neqns-1;
	for(n=1;n<=m;n++){
	    n1=n+1;
		n2=n+nbw-1;
		n3=(n-1)*nbw;
		if (n2>neqns) n2=neqns;
		for(i=n1;i<=n2;i++){
	        j=i-n+1;
            factor=mat[n3+j]/mat[n3+1];
			for(j=i;j<=n2;j++){
				ii=j-i+1;
				jj=j-n+1;
				mat[(i-1)*nbw+ii]-=factor*mat[n3+jj];
			}
		    rhs[i]-=factor*rhs[n];
		}
	}


	// Back substitution

	for(n=neqns;n>=2;n--){
		n1=n-1;
		n2=n-nbw+1;
        rhs[n]/=mat[(n-1)*nb+1];
		if (n2<1) n2=1;
		for(i=n1;i>=n2;i--){
			j=n-i+1;
			rhs[i]-=mat[(i-1)*nbw+j]*rhs[n];
		}
	}
	
	rhs[1]/=mat[1];
}

void file_out()
//writes the results on a file
{
	long i,n;
	n=npe*ne;
    
	ofstream out1;
	out1.open("out1.dat");
    for (i=1;i<=tndf;i++){
       	out1<<p[i]<<"  ";
    }
    
    ofstream out2;
	out2.open("out2.dat");
    for (i=1;i<=n;i++){
       	out2<<node[i]<<"  ";
    }
	
	ofstream out3;
	out3.open("out3.dat");
    for (i=1;i<=nn;i++){
       	out3<<xv[i]<<"  ";
    }

	ofstream out4;
	out4.open("out4.dat");
    for (i=1;i<=nn;i++){
       	out4<<yv[i]<<"  ";
    }

}